/**
 * \file: exchnd_backend.h
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * ExchndBackend - generic interface for an exception handler backend
 * An exception handler backend can store or otherwise output exception handler
 * messages. Different backends are created by filling below struct
 * with different function pointers.
 * Exception handler backends are organized in a forward-linked list,
 * allowing to call a function for all elements of the list.
 *
 * \component: exchndd
 *
 * \author: Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/


#ifndef _EXCHND_BACKEND_H
#define _EXCHND_BACKEND_H

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <pthread.h>
#include <linux/exchnd.h>
#include <sys/prctl.h>
#include "exchndd.h"

#define NO_BACKEND      0x0
#define ERRMEM_BACKEND  0x1
#define FILE_BACKEND    0x2
#define TRACE_BACKEND   0x4
#define DLT_BACKEND     0x8
#define ALL_BACKEND     0xf

#define DEFAULT_BACKEND ERRMEM_BACKEND

extern int backend_mask;

extern struct msg_list *mlist;

extern pthread_cond_t new_msg;
extern pthread_cond_t msg_handled;

typedef struct ExchndBackend {
    /* store - store an exception handler entry in the backend */
    unsigned int (*store)(struct ExchndBackend *, int len, char *);

    /* destroy - destroy the exception handler backend */
    void (*destroy)(struct ExchndBackend *);

    /* Thread of this backend */
    pthread_t BackendThread;

    /* exception handler backends are kept in a forward linked list */
    struct ExchndBackend *next;
} ExchndBackend_t;

/* exception handler backend functions */
/* Call each function for all elements in a forward-linked list */
/* of exception handler backends. */

/* exchnd_backend_store */
/* - Notify backends for a new incoming message. */
void exchnd_backend_store(char *, int);

void exchnd_log_error(const char *, ...);

/* exchnd_wait_default_backend */
/* - wait for default backend to store the message. */
void exchnd_wait_default_backend(void);
void exchnd_wake_backends(void);

void exchnd_set_file_name(char *);

int exchnd_init_storage(void);
void exchnd_cleanup_storage(void);

/* exchnd_backend_destroy */
/* - call destroy() on a list of exception handler backends */
static inline void exchnd_backend_destroy(ExchndBackend_t *backend)
{
    if (backend) {
        ExchndBackend_t *next = backend->next;

        if (backend->destroy)
            backend->destroy(backend);

        exchnd_backend_destroy(next);
    }
}

#ifdef DLT
ExchndBackend_t *exchnd_backend_create_dlt(ExchndBackend_t *);
#endif /* DLT */

ExchndBackend_t *exchnd_backend_create_file(ExchndBackend_t *, char *filename);

ExchndBackend_t *exchnd_backend_create_errmem(ExchndBackend_t *next);
#endif /* ifndef _EXCHND_BACKEND_H */